package geo

import (
	"context"
	"database/sql"
	"encoding/json"
	"fmt"
	"log"
	"os"

	"github.com/cloudwego/hertz/pkg/app"
	"github.com/cloudwego/hertz/pkg/common/utils"
	"github.com/cloudwego/hertz/pkg/protocol/consts"
	_ "github.com/lib/pq"
)

// Feature GeoJSON特征结构体
type Feature struct {
	Id         string                 `json:"id"`
	Type       string                 `json:"type"`
	Geometry   map[string]interface{} `json:"geometry"`
	Properties map[string]interface{} `json:"properties"`
}

// ProcessGeoJSONFile 处理GeoJSON文件的主函数
func ProcessGeoJSONFile(filename string) error {
	// 连接数据库
	db, err := connectDB()
	if err != nil {
		return fmt.Errorf("database connection failed: %v", err)
	}
	defer db.Close()

	// 创建表
	if err := createTable(db); err != nil {
		return fmt.Errorf("table creation failed: %v", err)
	}

	// 处理文件
	if err := processFile(filename, db); err != nil {
		return fmt.Errorf("file processing failed: %v", err)
	}

	return nil
}

func connectDB() (*sql.DB, error) {
	connStr := "postgres://liang:postgres@localhost/xpolt?sslmode=disable"
	return sql.Open("postgres", connStr)
}

func createTable(db *sql.DB) error {
	// _, err := db.Exec(`
	// 	CREATE TABLE IF NOT EXISTS geojson_data (
	// 		id SERIAL PRIMARY KEY,
	// 		type VARCHAR(64),
	// 		geom geometry(Geometry,4326),
	// 		properties JSONB
	// 	)
	// `)
	_, err := db.Exec(`
		create table IF NOT EXISTS opt_fly_routes (
			id serial PRIMARY key,
			type VARCHAR(64),
			properties JSONB,
			opt_time timestamp,
			geometry geometry(linestring,4326)
		);
	`)
	return err
}

func processFile(filename string, db *sql.DB) error {
	file, err := os.Open(filename)
	if err != nil {
		return err
	}
	defer file.Close()

	// 创建 JSON decoder
	decoder := json.NewDecoder(file)

	// 读取开始的 '{'
	if _, err := decoder.Token(); err != nil {
		return fmt.Errorf("读取文件开始标记失败: %v", err)
	}

	// 读取 "type" 字段
	for decoder.More() {
		token, err := decoder.Token()
		if err != nil {
			return fmt.Errorf("读取字段名失败: %v", err)
		}

		// 找到 "features" 数组开始
		if key, ok := token.(string); ok && key == "features" {
			// 读取数组开始标记 '['
			if _, err := decoder.Token(); err != nil {
				return fmt.Errorf("读取features数组开始标记失败: %v", err)
			}
			break
		}

		// 跳过其他字段的值（替换 Skip 方法）
		_, err = decoder.Token()
		if err != nil {
			return fmt.Errorf("跳过字段值失败: %v", err)
		}
	}

	// 开始事务
	tx, err := db.Begin()
	if err != nil {
		return err
	}
	defer tx.Rollback()

	count := 0
	// 逐个读取并处理特征
	for decoder.More() {
		var feature Feature
		if err := decoder.Decode(&feature); err != nil {
			log.Printf("解析特征失败: %v", err)
			continue
		}

		geometryJSON, err := json.Marshal(feature.Geometry)
		if err != nil {
			log.Printf("转换geometry失败: %v", err)
			continue
		}
		propertiesJSON, err := json.Marshal(feature.Properties)
		if err != nil {
			log.Printf("转换properties失败: %v", err)
			continue
		}

		// 	_, err = tx.Exec(`
		// 	INSERT INTO geojson_data (type, geom, properties)
		// 	VALUES ($1, ST_GeomFromGeoJSON($2), $3)
		// `, feature.Type, string(geometryJSON), string(propertiesJSON))

		_, err = tx.Exec(`
			INSERT INTO opt_fly_routes (type, geometry, properties,opt_time)
			VALUES ($1, ST_GeomFromGeoJSON($2), $3,now())
		`, feature.Type, string(geometryJSON), string(propertiesJSON))
		if err != nil {
			log.Printf("插入数据失败: %v", err)
			continue
		}

		count++
		if count%1000 == 0 { // 每1000条提交一次事务
			if err := tx.Commit(); err != nil {
				return err
			}
			tx, err = db.Begin() // 开始新的事务
			if err != nil {
				return err
			}
			log.Printf("已处理 %d 条记录", count)
		}
	}

	// 提交最后的事务
	if err := tx.Commit(); err != nil {
		return err
	}

	log.Printf("总共处理 %d 条记录", count)
	return nil
}

func HandleProcessGeoJSON(ctx context.Context, c *app.RequestContext) {
	var request struct {
		FilePath string `json:"filepath"`
	}

	if err := c.BindJSON(&request); err != nil {
		c.JSON(consts.StatusBadRequest, utils.H{"error": "invalid request body"})
		return
	}

	if request.FilePath == "" {
		c.JSON(consts.StatusBadRequest, utils.H{"error": "filepath is required"})
		return
	}

	err := ProcessGeoJSONFile(request.FilePath)
	if err != nil {
		c.JSON(consts.StatusInternalServerError, utils.H{"error": err.Error()})
		return
	}

	c.JSON(consts.StatusOK, utils.H{"message": "processing completed"})
}
