package data

import (
	"database/sql"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"strings"
	"time"

	"com.dliangx.xplot/appserver/model"
	geojson "github.com/paulmach/go.geojson"
)

type Msdata struct {
	ID                   int     `json:"id"`
	MemberLandId         int     `json:"member_land_id"`
	BlockCrop            string  `json:"block_crop"`
	BlockCropVariety     string  `json:"block_crop_variety"`
	IrrigationType       string  `json:"irrigation_type"`
	BlockArea            float64 `json:"block_area"`
	BlockCoordinates     string  `json:"block_coordinates"`
	EnclosureCoordinates string  `json:"enclosure_coordinates"`
	PreviewImageUrl      string  `json:"preview_image_url"`
	BlockId              int     `json:"block_id"`
	CreateTime           string  `json:"create_time"`
	UpdateTime           string  `json:"update_time"`
}

func (data Msdata) TransToOptArea() (area model.OptArea) {
	// 设置基本属性
	area.Properties = make(map[string]interface{})
	area.Properties["id"] = data.ID
	area.Properties["member_land_id"] = data.MemberLandId
	area.Properties["block_crop"] = data.BlockCrop
	area.Properties["block_crop_variety"] = data.BlockCropVariety
	area.Properties["irrigation_type"] = data.IrrigationType
	area.Properties["block_area"] = data.BlockArea
	area.Properties["block_coordinates"] = data.BlockCoordinates
	area.Properties["preview_image_url"] = data.PreviewImageUrl
	area.Properties["block_id"] = data.BlockId
	area.Properties["create_time"] = data.CreateTime

	// 设置类型
	area.Type = "polygon"

	// 解析更新时间
	ptime, err := time.Parse("2006-01-02 15:04:05", data.UpdateTime)
	if err != nil {
		fmt.Println("时间格式错误:", err)
	}
	area.OptTime = ptime

	// 解析围栏坐标为geometry
	geometry, err := data.parseEnclosureCoordinates()
	if err != nil {
		fmt.Println("解析围栏坐标错误:", err)
		area.Geometry = nil
	} else {
		area.Geometry = geometry
	}

	return area
}

func (data Msdata) parseEnclosureCoordinates() (*geojson.Geometry, error) {
	// 移除空格
	coordStr := strings.TrimSpace(data.EnclosureCoordinates)

	// 创建geometry结构
	geometry := &geojson.Geometry{
		Type: "Polygon",
	}

	// 解析坐标字符串
	var coordinates [][][]float64
	if err := json.Unmarshal([]byte(coordStr), &coordinates); err != nil {
		return nil, fmt.Errorf("解��坐标失败: %v", err)
	}

	// 设置坐标
	geometry.Polygon = coordinates

	return geometry, nil
}

// ProcessJsonData 处理JSON数据并写入数据库
func ProcessJsonData(jsonData []byte, db *sql.DB) error {
	// 开始事务
	tx, err := db.Begin()
	if err != nil {
		return fmt.Errorf("开始事务失败: %v", err)
	}
	defer tx.Rollback()

	// 解析JSON数据
	var msdataList []Msdata
	if err := json.Unmarshal(jsonData, &msdataList); err != nil {
		return fmt.Errorf("解析JSON失败: %v", err)
	}

	// 处理每条记录
	for _, msdata := range msdataList {
		optArea := msdata.TransToOptArea()

		// 将 Properties 和 Geometry 转换为 JSON
		propertiesJSON, err := json.Marshal(optArea.Properties)
		if err != nil {
			return fmt.Errorf("Properties转JSON失败: %v", err)
		}

		geometryJSON, err := json.Marshal(optArea.Geometry)
		if err != nil {
			return fmt.Errorf("Geometry转JSON失败: %v", err)
		}

		// 写入数据库
		_, err = tx.Exec(`
			INSERT INTO opt_areas (
				type, 
				properties,
				geometry,
				opt_time
			) VALUES ($1, $2, $3, $4)
		`,
			optArea.Type,
			propertiesJSON, // 使用JSON格式
			geometryJSON,   // 使用JSON格式
			optArea.OptTime,
		)

		if err != nil {
			return fmt.Errorf("插入数据失败: %v", err)
		}
	}

	// 提交事务
	if err := tx.Commit(); err != nil {
		return fmt.Errorf("提交事务失败: %v", err)
	}

	return nil
}

// ProcessJsonFile 从文件读取并处理JSON数据
func ProcessJsonFile(filePath string, db *sql.DB) error {
	// 读取JSON文件
	jsonData, err := os.ReadFile(filePath)
	if err != nil {
		return fmt.Errorf("读取JSON文件失败: %v", err)
	}

	// 使用已有的ProcessJsonData处理数据
	if err := ProcessJsonData(jsonData, db); err != nil {
		return fmt.Errorf("处理数据失败: %v", err)
	}

	log.Printf("成功处理文件: %s", filePath)
	return nil
}
