package geo

import (
	"database/sql"
	"fmt"

	"com.dliangx.xplot/appserver/model"
)

func SaveRoute(line model.GeoJson) (model.OptRoute, error) {
	// 连接数据库
	db, err := connectDB()
	if err != nil {
		return model.OptRoute{}, fmt.Errorf("database connection failed: %v", err)
	}
	defer db.Close()

	// 开启事务
	tx, err := db.Begin()
	if err != nil {
		return model.OptRoute{}, fmt.Errorf("begin transaction failed: %w", err)
	}
	defer tx.Rollback()

	// 插入数据
	var route model.OptRoute
	if line.Opt == 1 {
		err = tx.QueryRow(`
		INSERT INTO opt_routes (type, geometry, properties, opt_time)
		VALUES ($1, ST_GeomFromGeoJSON($2), $3, now())
		RETURNING id, type, properties, opt_time
	`, line.Type, line.Geometry, line.Properties).Scan(
			&route.ID, &route.Type, &route.Properties, &route.OptTime,
		)
	} else if line.Opt == 2 {
		err = tx.QueryRow(`
		INSERT INTO opt_fly_routes (type, geometry, properties, opt_time)
		VALUES ($1, ST_GeomFromGeoJSON($2), $3, now())
		RETURNING id, type, properties, opt_time
	`, line.Type, line.Geometry, line.Properties).Scan(
			&route.ID, &route.Type, &route.Properties, &route.OptTime,
		)
	}

	if err != nil {
		return model.OptRoute{}, fmt.Errorf("insert route failed: %w", err)
	}

	// 提交事务
	if err = tx.Commit(); err != nil {
		return model.OptRoute{}, fmt.Errorf("commit transaction failed: %w", err)
	}

	return route, nil
}

func SaveOptArea(optArea model.GeoJson) (model.OptArea, error) {
	db, err := connectDB()
	if err != nil {
		return model.OptArea{}, fmt.Errorf("database connection failed: %v", err)
	}
	defer db.Close()

	tx, err := db.Begin()
	if err != nil {
		return model.OptArea{}, fmt.Errorf("begin transaction failed: %w", err)
	}
	defer tx.Rollback()

	var area model.OptArea
	err = tx.QueryRow(`
		INSERT INTO opt_areas (type, geometry, properties, opt_time)
		VALUES ($1, ST_GeomFromGeoJSON($2), $3, now())
		RETURNING id, type, properties, opt_time
	`, optArea.Type, optArea.Geometry, optArea.Properties).Scan(
		&area.ID, &area.Type, &area.Properties, &area.OptTime,
	)
	if err != nil {
		return model.OptArea{}, fmt.Errorf("insert area failed: %w", err)
	}

	if err = tx.Commit(); err != nil {
		return model.OptArea{}, fmt.Errorf("commit transaction failed: %w", err)
	}

	return area, nil
}

func UpdateRoute(line model.GeoJson) (model.OptRoute, error) {
	db, err := connectDB()
	if err != nil {
		return model.OptRoute{}, fmt.Errorf("database connection failed: %v", err)
	}
	defer db.Close()

	tx, err := db.Begin()
	if err != nil {
		return model.OptRoute{}, fmt.Errorf("begin transaction failed: %w", err)
	}
	defer tx.Rollback()

	var route model.OptRoute
	if line.Opt == 1 {
		err = tx.QueryRow(`
		UPDATE opt_routes 
		SET type = $1, geometry = ST_GeomFromGeoJSON($2), properties = $3, opt_time = now()
		WHERE id = $4
		RETURNING id, type, properties, opt_time
	`, line.Type, line.Geometry, line.Properties, line.ID).Scan(
			&route.ID, &route.Type, &route.Properties, &route.OptTime,
		)
	} else if line.Opt == 2 {
		err = tx.QueryRow(`
		UPDATE opt_fly_routes 
		SET type = $1, geometry = ST_GeomFromGeoJSON($2), properties = $3, opt_time = now()
		WHERE id = $4
		RETURNING id, type, properties, opt_time
	`, line.Type, line.Geometry, line.Properties, line.ID).Scan(
			&route.ID, &route.Type, &route.Properties, &route.OptTime,
		)
	}

	if err != nil {
		return model.OptRoute{}, fmt.Errorf("update route failed: %w", err)
	}

	if err = tx.Commit(); err != nil {
		return model.OptRoute{}, fmt.Errorf("commit transaction failed: %w", err)
	}

	return route, nil
}

func UpdateOptArea(optArea model.GeoJson) (model.OptArea, error) {
	db, err := connectDB()
	if err != nil {
		return model.OptArea{}, fmt.Errorf("database connection failed: %v", err)
	}
	defer db.Close()

	tx, err := db.Begin()
	if err != nil {
		return model.OptArea{}, fmt.Errorf("begin transaction failed: %w", err)
	}
	defer tx.Rollback()

	var area model.OptArea
	err = tx.QueryRow(`
		UPDATE opt_areas 
		SET type = $1, geometry = ST_GeomFromGeoJSON($2), properties = $3, opt_time = now()
		WHERE id = $4
		RETURNING id, type, properties, opt_time
	`, optArea.Type, optArea.Geometry, optArea.Properties, optArea.ID).Scan(
		&area.ID, &area.Type, &area.Properties, &area.OptTime,
	)
	if err != nil {
		return model.OptArea{}, fmt.Errorf("update area failed: %w", err)
	}

	if err = tx.Commit(); err != nil {
		return model.OptArea{}, fmt.Errorf("commit transaction failed: %w", err)
	}

	return area, nil
}

func GetRoute(id int, opt_type int) (model.OptRoute, error) {
	db, err := connectDB()
	if err != nil {
		return model.OptRoute{}, fmt.Errorf("database connection failed: %v", err)
	}
	defer db.Close()

	var route model.OptRoute
	var query string
	if opt_type == 1 {
		query = `
			SELECT id, type, properties, opt_time, ST_AsGeoJSON(geometry) as geometry
			FROM opt_routes WHERE id = $1`
	} else if opt_type == 2 {
		query = `
			SELECT id, type, properties, opt_time, ST_AsGeoJSON(geometry) as geometry
			FROM opt_fly_routes WHERE id = $1`
	}

	err = db.QueryRow(query, id).Scan(
		&route.ID, &route.Type, &route.Properties, &route.OptTime, &route.Geometry,
	)
	if err != nil {
		if err == sql.ErrNoRows {
			return model.OptRoute{}, fmt.Errorf("route not found")
		}
		return model.OptRoute{}, fmt.Errorf("query route failed: %w", err)
	}

	return route, nil
}

func GetOptArea(id int, opt_type int) (model.OptArea, error) {
	db, err := connectDB()
	if err != nil {
		return model.OptArea{}, fmt.Errorf("database connection failed: %v", err)
	}
	defer db.Close()

	var area model.OptArea
	err = db.QueryRow(`
		SELECT id, type, properties, opt_time, ST_AsGeoJSON(geometry) as geometry
		FROM opt_areas WHERE id = $1
	`, id).Scan(
		&area.ID, &area.Type, &area.Properties, &area.OptTime, &area.Geometry,
	)
	if err != nil {
		if err == sql.ErrNoRows {
			return model.OptArea{}, fmt.Errorf("area not found")
		}
		return model.OptArea{}, fmt.Errorf("query area failed: %w", err)
	}

	return area, nil
}

func DeleteRoute(id int, opt_type int) (bool, error) {
	db, err := connectDB()
	if err != nil {
		return false, fmt.Errorf("database connection failed: %v", err)
	}
	defer db.Close()

	var result sql.Result
	if opt_type == 1 {
		result, err = db.Exec(`DELETE FROM opt_routes WHERE id = $1`, id)
	} else if opt_type == 2 {
		result, err = db.Exec(`DELETE FROM opt_fly_routes WHERE id = $1`, id)
	}

	if err != nil {
		return false, fmt.Errorf("delete route failed: %w", err)
	}

	rowsAffected, err := result.RowsAffected()
	if err != nil {
		return false, fmt.Errorf("get affected rows failed: %w", err)
	}

	return rowsAffected > 0, nil
}

func DeleteOptArea(id int, opt_type int) (bool, error) {
	db, err := connectDB()
	if err != nil {
		return false, fmt.Errorf("database connection failed: %v", err)
	}
	defer db.Close()

	var result sql.Result
	result, err = db.Exec(`DELETE FROM opt_area WHERE id = $1`, id)

	if err != nil {
		return false, fmt.Errorf("delete route failed: %w", err)
	}

	rowsAffected, err := result.RowsAffected()
	if err != nil {
		return false, fmt.Errorf("get affected rows failed: %w", err)
	}

	return rowsAffected > 0, nil
}

func GetRelatedRoute(obj model.GeoJson) ([]model.OptArea, error) {

	return []model.OptArea{}, nil
}

func GetRelatedOptArea(obj model.GeoJson) ([]model.OptArea, error) {

	return []model.OptArea{}, nil
}

func GetRelatedAdmArea(obj model.GeoJson) ([]model.GeoJson, error) {

	return []model.GeoJson{}, nil
}

func CheckOptAreasCrossById(id1, id2 int) (bool, error) {

	return false, nil
}

func CheckAreasCross(obj1, obj2 model.GeoJson) (bool, error) {

	return false, nil
}
