# API接口集成文档

## 📋 文档说明

本文档面向需要集成**浦发银行农链通接口系统**的外部开发团队，详细说明如何调用本系统提供的API接口。

## 🏗️ 系统架构说明

### 角色定位
- **本系统**: 接口服务提供者，负责业务流程管理和状态查询
- **银行系统**: 接口调用方，通过API获取业务服务
- **业务平台**: 接口调用方，管理贷款业务流程
- **前端页面**: 通过JavaScript调用部分接口

### 集成模式
```mermaid
%%{init: {'theme':'base', 'themeVariables': {'background': '#ffffff', 'primaryColor': '#fff', 'primaryTextColor': '#000', 'primaryBorderColor': '#000', 'lineColor': '#000'}}}%%
graph TD
    A[银行核心系统] --> D[本系统API]
    B[业务管理平台] --> D
    C[第三方集成系统] --> D
    
    D --> E[身份认证]
    D --> F[支付状态查询]
    D --> G[URL跳转服务]
    
    H[前端页面] --> G
    
    style D fill:#e3f2fd,stroke:#1976d2
    style A fill:#fff8e1,stroke:#f57c00
    style B fill:#fff8e1,stroke:#f57c00
    style C fill:#fff8e1,stroke:#f57c00
```

## 🔌 API接口清单

### 接口总览

| 接口名称 | 方法 | 路径 | 功能描述 | 调用方 |
|----------|------|------|----------|--------|
| 身份认证接口 | POST | `/nlt/pufa/auth` | 用户身份验证和贷款申请处理 | 银行系统/业务平台 |
| 支付状态查询 | POST | `/nlt/pufa/queryPayResult` | 查询担保费支付状态 | 银行系统/业务平台 |
| 跳转URL查询 | POST | `/nlt/pufa/queryJumpUrl` | 获取业务流程跳转链接 | 前端页面/系统 |

## 🔐 安全要求

### 1. 国密SM2加密

**所有接口均采用国密SM2算法进行端到端加密**

#### 加密流程
```mermaid
%%{init: {'theme':'base', 'themeVariables': {'background': '#ffffff', 'primaryColor': '#fff', 'primaryTextColor': '#000', 'primaryBorderColor': '#000', 'lineColor': '#000'}}}%%
sequenceDiagram
    participant Client as 调用方
    participant API as 接口系统
    
    Note over Client: 1. 准备请求数据
    Client->>Client: 2. SM2公钥加密body
    Client->>Client: 3. SM2私钥签名整个请求
    Client->>API: 4. 发送加密请求
    API->>API: 5. 验证签名
    API->>API: 6. SM2私钥解密body
    API->>API: 7. 处理业务逻辑
    API->>API: 8. SM2公钥加密响应
    API->>API: 9. SM2私钥签名响应
    API->>Client: 10. 返回加密响应
    Client->>Client: 11. 验证签名并解密
```

#### 密钥配置

**调用方需要配置的密钥信息**:
```json
{
  "本系统公钥": "04A7CD09260A67113F988F530154AD6A70B2A4DD3E00BD27BB124E7E7051FC0C97E7AC3C5A6CB6C9BB459BEF252761AD1AE727718498CA3130D67CFC84F9B1BB1F",
  "银行测试私钥": "308a6311076aa18ec591ce0b300c6f92b1e58438d0ab0962d67b7d163b0e2f8d",
  "银行测试公钥": "041f41683ee8d5204958db303b16c97a912b1d2a7ee640f767001395ccfcc16c48e9e81ed2b0e540c53a0836040665ef98f2488ee4cc7e9c525d8ecc92e42d62f1"
}
```

### 2. 请求响应格式

#### 统一请求格式
```json
{
  "request": {
    "head": {
      "requestTime": "yyyyMMddHHmmss",
      "versionId": "V1.0",
      "serviceId": "服务标识",
      "serviceSn": "请求流水号",
      "channelId": "渠道标识",
      "sid": "会话标识",
      "businessChannel": "业务渠道"
    },
    "body": "SM2加密后的业务数据(Hex编码)"
  },
  "signature": "SM2数字签名(Hex编码)"
}
```

#### 统一响应格式
```json
{
  "response": {
    "head": {
      "code": "响应码",
      "serviceTime": "yyyyMMddHHmmss",
      "serviceSn": "请求流水号"
    },
    "body": "SM2加密后的响应数据(Hex编码)"
  },
  "signature": "SM2数字签名(Hex编码)"
}
```

## 📝 接口详细规范

### 1. 身份认证接口

#### 接口信息
- **路径**: `POST /nlt/pufa/auth`
- **功能**: 处理用户身份认证和贷款申请
- **调用场景**: 用户提交贷款申请时

#### 请求参数 (解密后的body内容)
```json
{
  "cellPhone": "手机号码",
  "userName": "用户姓名", 
  "idNo": "身份证号",
  "addr": "地址",
  "loanAmt": "申请金额(分)",
  "term": "贷款期限(月)",
  "duebillNoOrg": "外部流水号",
  "jumURL": "跳转URL"
}
```

#### 响应数据 (解密后的body内容)
```json
{
  "payOrderNo": "支付订单号",
  "directURL": "直接跳转URL"
}
```

#### 业务逻辑
1. 验证身份证号和姓名匹配
2. 将申请金额从分转换为元存储
3. 更新订单信息到数据库
4. 返回担保费缴纳页面URL

#### 调用示例
```bash
curl -X POST https://api.example.com/nlt/pufa/auth \
  -H "Content-Type: application/json" \
  -d '{
    "request": {
      "head": {
        "requestTime": "20240101120000",
        "versionId": "V1.0", 
        "serviceId": "pfyhcxzfjg",
        "serviceSn": "202401011200001",
        "channelId": "BANK001",
        "sid": "session_202401011200001",
        "businessChannel": "online"
      },
      "body": "SM2加密后的Hex字符串"
    },
    "signature": "SM2签名后的Hex字符串"
  }'
```

### 2. 支付状态查询接口

#### 接口信息
- **路径**: `POST /nlt/pufa/queryPayResult`
- **功能**: 查询担保费支付状态
- **调用场景**: 检查用户是否已完成担保费缴纳

#### 请求参数 (解密后的body内容)
```json
{
  "duebillNoOrg": "外部流水号"
}
```

#### 响应数据 (解密后的body内容)
```json
{
  "payOrderNo": "支付订单号",
  "duebillNoOrg": "外部流水号", 
  "status": "SUCCESS/FAIL",
  "orderTime": "订单时间",
  "failMsg": "失败原因"
}
```

#### 业务规则
- 查询条件: `status = 37 AND amount_r <= quota_r`
- 状态转换: 数据库status=37 → 返回"SUCCESS"
- 额度校验: 申请金额必须小于等于预授信额度

### 3. 跳转URL查询接口

#### 接口信息
- **路径**: `POST /nlt/pufa/queryJumpUrl`
- **功能**: 获取业务流程跳转链接
- **调用场景**: 担保费缴纳完成后获取下一步链接

#### 请求参数 (非加密，直接JSON)
```json
{
  "duebillNoOrg": "外部流水号"
}
```

#### 响应数据 (非加密，直接JSON)
```json
{
  "head": {
    "code": "000000",
    "serviceTime": "20240101120000"
  },
  "body": "https://next-step-url.com"
}
```

## 🔧 集成技术要求

### 1. 开发环境

#### 推荐技术栈
- **编程语言**: Java/Go/Python/C#等
- **加密库**: 支持国密SM2算法的加密库
- **HTTP客户端**: 支持POST请求和JSON处理

#### 国密SM2算法库推荐
| 语言 | 推荐库 | 说明 |
|------|--------|------|
| Java | BC (Bouncy Castle) | 支持国密算法 |
| Go | github.com/tjfoc/gmsm | 专门的国密算法库 |
| Python | gmssl | 国密SSL/TLS协议库 |
| .NET | GmSSL.NET | .NET平台国密实现 |

### 2. 集成步骤

#### Step 1: 环境准备
1. 安装国密SM2算法库
2. 配置密钥对(公钥用于加密，私钥用于签名)
3. 准备HTTP客户端工具

#### Step 2: 实现加密解密
```python
# Python示例代码框架
def encrypt_request(data, public_key):
    """使用SM2公钥加密请求数据"""
    # 实现SM2加密逻辑
    pass

def sign_request(data, private_key):
    """使用SM2私钥签名请求"""
    # 实现SM2签名逻辑  
    pass

def call_api(endpoint, business_data):
    """调用API接口"""
    # 1. 构造请求头
    head = {
        "requestTime": datetime.now().strftime("%Y%m%d%H%M%S"),
        "versionId": "V1.0",
        "serviceId": "your_service_id",
        "serviceSn": generate_serial_number(),
        "channelId": "your_channel_id",
        "sid": generate_session_id(),
        "businessChannel": "online"
    }
    
    # 2. 加密业务数据
    encrypted_body = encrypt_request(business_data, public_key)
    
    # 3. 构造请求
    request_data = {
        "request": {
            "head": head,
            "body": encrypted_body
        }
    }
    
    # 4. 签名请求
    signature = sign_request(request_data["request"], private_key)
    request_data["signature"] = signature
    
    # 5. 发送HTTP请求
    response = requests.post(endpoint, json=request_data)
    
    # 6. 处理响应
    return process_response(response)
```

#### Step 3: 错误处理
```python
def process_response(response):
    """处理API响应"""
    if response.status_code != 200:
        raise APIException(f"HTTP错误: {response.status_code}")
    
    data = response.json()
    code = data.get("response", {}).get("head", {}).get("code")
    
    if code == "000000":
        # 成功，解密响应数据
        return decrypt_response(data)
    else:
        # 业务错误
        error_msg = get_error_message(code)
        raise BusinessException(f"业务错误[{code}]: {error_msg}")
```

## 📊 错误码说明

| 错误码 | 错误描述 | 处理建议 |
|--------|----------|----------|
| 000000 | 成功 | 正常处理 |
| W020103 | 查询无数据 | 检查流水号是否正确 |
| W140001 | 请求参数为空 | 检查必填参数 |
| W140003 | 请求参数格式不正确 | 验证参数格式 |
| W149999 | 系统异常 | 稍后重试或联系技术支持 |
| W380002 | 申请人信息比对不一致 | 检查姓名和身份证号匹配 |
| W380005 | 验签错误 | 检查签名算法和密钥配置 |
| W380006 | 解密错误 | 检查加密算法和密钥配置 |

## 🧪 测试指南

### 1. 接口测试环境

#### 测试环境信息
```
测试域名: https://test.nlt-api.example.com
端口: 8970
协议: HTTPS
```

#### 测试账号
```
测试用户: 张三
身份证号: 420982198604026032  
手机号: 13811111111
```

### 2. 测试用例

#### 用例1: 身份认证成功
```json
{
  "cellPhone": "13811111111",
  "userName": "张三",
  "idNo": "420982198604026032",
  "addr": "湖北省武汉市",
  "loanAmt": "1000000",
  "term": "12", 
  "duebillNoOrg": "TEST202401011200001",
  "jumURL": "https://bank.example.com/next-step"
}
```

#### 用例2: 支付状态查询
```json
{
  "duebillNoOrg": "TEST202401011200001"
}
```

### 3. 常见问题

#### Q1: 加密后的数据过长怎么办？
A: SM2加密会增加数据长度，建议：
- 使用gzip压缩原始数据
- 采用分段加密处理大数据

#### Q2: 签名验证失败如何排查？
A: 检查顺序：
1. 确认使用正确的私钥
2. 验证JSON序列化顺序
3. 检查字符编码(建议UTF-8)
4. 确认签名算法参数

#### Q3: 如何处理网络超时？
A: 建议配置：
- 连接超时: 10秒
- 读取超时: 30秒
- 重试机制: 3次，间隔2秒

## 📞 技术支持

### 联系方式
- **技术支持**: [tech-support@example.com]
- **API文档**: [https://docs.nlt-api.example.com]
- **问题反馈**: [issues@example.com]

### 支持时间
- **工作日**: 9:00 - 18:00
- **紧急问题**: 7x24小时支持

---

**文档版本**: v1.0  
**最后更新**: 2024年  
**维护团队**: 接口集成支持组 