# 浦发银行农链通接口系统 - 业务设计文档

## 📋 文档概述

本文档详细说明了浦发银行农链通接口系统的核心业务逻辑，包括担保费支付流程、预授信额度控制机制和风险管控策略。

## 💰 担保费支付业务逻辑

### 🔄 核心业务流程

#### 1. 状态定义与含义

| 状态值 | 业务含义 | 说明 |
|--------|----------|------|
| **37** | 担保费已支付/审批通过 | 系统中表示支付成功的关键标识符 |
| 其他状态 | 担保费未支付/处理中 | 表示支付流程尚未完成 |

#### 2. 完整业务流程

```mermaid
%%{init: {'theme':'base', 'themeVariables': {'background': '#ffffff', 'primaryColor': '#fff', 'primaryTextColor': '#000', 'primaryBorderColor': '#000', 'lineColor': '#000'}}}%%
graph TD
    A[用户身份认证] --> B[认证成功]
    B --> C[返回担保费缴纳页面URL]
    C --> D[用户访问担保费页面]
    D --> E[显示：请线下缴纳担保费]
    E --> F[用户线下缴纳担保费]
    F --> G[银行/机构更新订单状态为37]
    G --> H[用户点击确定按钮]
    H --> I[调用queryJumpUrl获取下一步链接]
    I --> J[跳转到业务页面]
    
    K[外部系统查询] --> L[调用queryPayResult接口]
    L --> M{状态是否为37?}
    M -->|是| N[返回SUCCESS]
    M -->|否| O[返回FAIL]
    
    style E fill:#fff2cc,stroke:#d6b656
    style G fill:#d5e8d4,stroke:#82b366
    style M fill:#f8cecc,stroke:#b85450
```

#### 3. 关键代码实现

**支付状态查询逻辑**:
```go
// 查询条件：状态为37且实际金额小于等于预授信额度
rows, err := db.DB.Query("select serial_no,status,createtime,reject_cause from loan_orderinfo where serial_no = ? and status = 37 and amount_r <= quota_r order by createtime", &req.Request.DuebillNoOrg)

// 状态转换逻辑
if resp.Response.Status == "37" {
    resp.Response.Status = "SUCCESS"  // 担保费已支付
} else {
    resp.Response.Status = "FAIL"     // 担保费未支付
}
```

**前端交互处理**:
```javascript
// 用户确认担保费缴纳后的跳转逻辑
document.getElementById("confirmButton").onclick = function () {
    fetch("https://tknlt.agrolinking.cn/api/nlt/pufa/queryJumpUrl", {
        method: "POST",
        body: JSON.stringify({duebillNoOrg: idParam})
    }).then(response => response.json())
      .then(res => {
          if (res.head.code == "000000") {
              window.location.href = res.body;  // 跳转到下一业务环节
          }
      });
};
```

## 💳 预授信额度控制机制

### 🔍 核心对比逻辑

系统实现了严格的**预授信额度控制**，确保用户申请的贷款金额不超过其预设的信用额度。

#### 1. 字段定义

| 字段名 | 数据类型 | 业务含义 | 数据来源 |
|--------|----------|----------|----------|
| `amount_r` | DECIMAL | **实际申请贷款金额** | 用户申请时提交，系统验证后存储 |
| `quota_r` | DECIMAL | **预授信额度** | 银行风控系统预先设定的最大可贷金额 |

#### 2. 额度判断条件

**核心规则**: `amount_r <= quota_r`

只有当**实际申请金额**小于等于**预授信额度**时，系统才认为该订单符合放贷条件。

#### 3. 金额处理流程

```mermaid
%%{init: {'theme':'base', 'themeVariables': {'background': '#ffffff', 'primaryColor': '#fff', 'primaryTextColor': '#000', 'primaryBorderColor': '#000', 'lineColor': '#000'}}}%%
graph TD
    A[用户提交申请金额] --> B[系统接收LoanAmt]
    B --> C[金额单位转换: 分→元]
    C --> D[更新amount_r字段]
    D --> E[支付完成: status=37]
    E --> F[查询时验证]
    F --> G{amount_r <= quota_r?}
    G -->|是| H[返回SUCCESS - 符合授信]
    G -->|否| I[不返回记录 - 超出授信]
    
    style C fill:#fff2cc,stroke:#d6b656
    style G fill:#f8cecc,stroke:#b85450
    style H fill:#d5e8d4,stroke:#82b366
    style I fill:#ffcccc,stroke:#ff6666
```

#### 4. 代码实现

**金额转换与存储**:
```go
amt := req.Request.LoanAmt
// 从分转换为元 (金融系统常见做法，避免浮点数精度问题)
float32Value, err := strconv.ParseFloat(amt, 32)
value := float32Value / 100

// 更新实际贷款金额
_, err = db.DB.Exec("update loan_orderinfo set serial_no = ?,disburse_sign_url=?,amount_r=?,term_r=? where order_sn=? ",
    &req.Request.DuebillNoOrg, &req.Request.JumURL, &value, &req.Request.Term, &ordersn)
```

## 🛡️ 风险控制策略

### 1. 多层次风控机制

| 控制层级 | 控制点 | 实现方式 | 风险防范 |
|----------|--------|----------|----------|
| **申请阶段** | 身份验证 | 身份证号+姓名匹配 | 防止冒名申请 |
| **额度阶段** | 预授信控制 | amount_r <= quota_r | 防止超额放贷 |
| **支付阶段** | 状态管控 | status = 37 | 确保担保费已缴纳 |
| **查询阶段** | 综合验证 | 多条件AND查询 | 多重校验确保合规 |

### 2. 业务合规性保障

**查询条件组合**:
```sql
SELECT serial_no, status, createtime, reject_cause 
FROM loan_orderinfo 
WHERE serial_no = ? 
  AND status = 37 
  AND amount_r <= quota_r 
ORDER BY createtime
```

**三重验证机制**:
1. **流水号匹配**: 确保查询的是正确订单
2. **支付状态确认**: 必须是已支付状态(37)
3. **额度合规验证**: 申请金额不能超过预授信额度

### 3. 异常处理策略

| 异常情况 | 系统响应 | 业务处理 |
|----------|----------|----------|
| 超出预授信额度 | 不返回查询结果 | 提示额度不足，建议重新申请 |
| 担保费未支付 | 返回FAIL状态 | 引导用户完成担保费缴纳 |
| 订单不存在 | 返回NODATA | 提示订单信息错误 |

## 📊 系统架构角色分工

### 1. 角色职责划分

| 系统角色 | 主要职责 | 核心功能 |
|----------|----------|----------|
| **本接口系统** | 流程引导与状态查询 | • 身份认证<br>• 担保费缴纳页面<br>• 支付状态查询<br>• 业务流程跳转 |
| **银行/支付机构** | 担保费收取与状态更新 | • 线下担保费处理<br>• 订单状态更新(status=37)<br>• 资金清算 |
| **风控系统** | 额度管理与风险评估 | • 预授信额度设定(quota_r)<br>• 用户信用评估<br>• 风险监控 |
| **业务系统** | 贷款流程管理 | • 贷款申请处理<br>• 状态监控<br>• 业务流程控制 |

## 🔌 外部接口集成关系

### 1. 接口调用方向说明

**重要说明**: 本系统是**接口服务提供者**，不主动调用银行接口，而是**被外部系统调用**。

#### 1.1 本系统对外提供的接口

| 接口名称 | 调用方 | 接口路径 | 用途 |
|----------|--------|----------|------|
| **身份认证接口** | 银行系统/业务平台 | `POST /nlt/pufa/auth` | 处理用户身份验证和贷款申请 |
| **支付状态查询** | 银行系统/业务平台 | `POST /nlt/pufa/queryPayResult` | 查询担保费支付状态 |
| **跳转URL查询** | 前端页面 | `POST /nlt/pufa/queryJumpUrl` | 获取业务流程跳转链接 |

#### 1.2 外部系统调用本系统的场景

```mermaid
%%{init: {'theme':'base', 'themeVariables': {'background': '#ffffff', 'primaryColor': '#fff', 'primaryTextColor': '#000', 'primaryBorderColor': '#000', 'lineColor': '#000'}}}%%
graph LR
    A[银行核心系统] --> B[本接口系统]
    C[业务管理平台] --> B
    D[前端页面] --> B
    E[第三方系统] --> B
    
    B --> F[数据库]
    G[银行线下系统] --> F
    
    style B fill:#e1f5fe,stroke:#0277bd
    style A fill:#fff3e0,stroke:#f57c00
    style C fill:#fff3e0,stroke:#f57c00
    style E fill:#fff3e0,stroke:#f57c00
    style G fill:#ffebee,stroke:#d32f2f
```

#### 1.3 数据更新机制

**担保费状态更新方式**:
- ❌ **非接口调用**: 银行不通过API接口更新状态
- ✅ **直接数据库操作**: 银行线下处理后直接更新数据库字段
  - 更新 `status = 37` (担保费已支付)
  - 系统通过查询接口返回最新状态

### 2. 系统集成架构

#### 2.1 实际的系统调用关系

| 调用方向 | 说明 | 实现方式 |
|----------|------|----------|
| **外部 → 本系统** | 银行等外部系统调用本系统接口 | HTTP POST + 国密SM2加密 |
| **本系统 → 数据库** | 查询和更新业务数据 | MySQL数据库连接 |
| **银行线下 → 数据库** | 担保费处理后更新状态 | 直接数据库操作 |
| **前端 → 本系统** | 页面获取跳转URL | JavaScript fetch调用 |

#### 2.2 不存在的调用关系

❌ **本系统不会调用以下接口**:
- 银行支付接口
- 银行账户查询接口  
- 银行资金划转接口
- 第三方支付平台接口

**原因**: 本系统定位为**业务流程管理和状态查询服务**，担保费支付采用**线下处理模式**。

### 2. 数据流转图

```mermaid
%%{init: {'theme':'base', 'themeVariables': {'background': '#ffffff', 'primaryColor': '#fff', 'primaryTextColor': '#000', 'primaryBorderColor': '#000', 'lineColor': '#000'}}}%%
sequenceDiagram
    participant U as 用户
    participant API as 接口系统
    participant DB as 数据库
    participant Bank as 银行系统
    participant Risk as 风控系统

    Note over Risk: 预设quota_r额度
    U->>API: 1. 提交贷款申请
    API->>DB: 2. 验证身份&更新amount_r
    API-->>U: 3. 返回担保费页面URL
    
    U->>Bank: 4. 线下缴纳担保费
    Bank->>DB: 5. 更新status=37
    
    U->>API: 6. 查询支付状态
    API->>DB: 7. 验证: status=37 AND amount_r<=quota_r
    DB-->>API: 8. 返回符合条件的记录
    API-->>U: 9. 返回SUCCESS/FAIL
```

## 🔧 技术实现要点

### 1. 数据精度处理

**金融金额处理最佳实践**:
- 前端传输：以"分"为单位，避免浮点数传输精度问题
- 后端存储：转换为"元"，使用DECIMAL类型确保精度
- 业务计算：统一单位，避免精度丢失

### 2. 状态机设计

订单状态遵循严格的状态机转换：
```
初始状态 → 审核中 → 担保费待缴 → 担保费已缴(37) → 放款成功
```

### 3. 安全性考虑

- **国密加密**: 使用SM2算法保护敏感数据传输
- **数字签名**: 确保数据完整性和身份认证
- **多重验证**: 身份、额度、支付状态的多层验证
- **日志记录**: 完整的操作审计日志

## 📈 业务指标监控

### 1. 关键业务指标

| 指标类型 | 监控指标 | 业务意义 |
|----------|----------|----------|
| **风控指标** | 超额申请率 | amount_r > quota_r 的申请占比 |
| **支付指标** | 担保费缴纳率 | status=37的订单占比 |
| **效率指标** | 平均处理时长 | 从申请到放款的时间 |
| **质量指标** | 异常订单率 | 各类异常情况的发生频率 |

### 2. 预警机制

- **额度预警**: 当超额申请率超过阈值时触发预警
- **支付预警**: 担保费缴纳超时自动提醒
- **系统预警**: API调用异常或响应超时告警

---

**文档版本**: v1.0  
**最后更新**: 2024年  
**维护团队**: 农链通系统开发组 