package model

import (
	"crypto/rand"
	"encoding/json"
	"log"
	"strings"

	"github.com/tjfoc/gmsm/sm2"
	"nlt.com/pf/nltconst"
)

type CryptHttpBodyReq[T any] struct {
	Request   HttpBodyReq[T]
	Signature string
}

type CryptHttpBodyResp[T any] struct {
	Response  HttpBodyResp[T]
	Signature string
}

type HttpBodyReq[T any] struct {
	Head    ReqHead `json:"head"`
	Request T       `json:"body"`
}

type HttpBodyResp[T any] struct {
	Head     RespHead `json:"head"`
	Response T        `json:"body"`
}

type ReqHead struct {
	RequestTime string `json:"requestTime"`
	ServiceSn   string `json:"serviceSn"`
}

type RespHead struct {
	Code        string `json:"code"`
	ServiceTime string `json:"serviceTime"`
	ServiceSn   string `json:"serviceSn"`
}

func EncryptAndSign[T any](resp HttpBodyResp[T]) CryptHttpBodyResp[string] {
	var cresp CryptHttpBodyResp[string]
	privateKeyBytes, _ := sm2.GenerateKey(strings.NewReader(nltconst.SM2_PRIVATE_KEY))

	// 对应的公钥
	publicKey := &privateKeyBytes.PublicKey
	body, err := json.Marshal(resp.Response)
	if err != nil {
		log.Println(err.Error())
	}
	ciphertext, err := sm2.Encrypt(publicKey, body, rand.Reader, sm2.C1C2C3)
	if err != nil {
		log.Println(err)
	}
	cresp.Response.Head = resp.Head
	cresp.Response.Response = string(ciphertext)

	response, err := json.Marshal(resp)
	uid := []byte("tk")
	r, s, err := sm2.Sm2Sign(privateKeyBytes, response, uid, rand.Reader)
	if err != nil {
		log.Println(err)
	}
	rBytes := r.Bytes()
	sBytes := s.Bytes()

	signature := append(rBytes, sBytes...)
	cresp.Signature = string(signature)
	return cresp
}

func VerifyAndDecrypt[T any](creq CryptHttpBodyReq[string]) HttpBodyReq[T] {
	var req HttpBodyReq[T]
	privateKeyBytes, _ := sm2.GenerateKey(strings.NewReader(nltconst.SM2_PRIVATE_KEY))
	signature := creq.Signature
	r, s, err := sm2.SignDataToSignDigit([]byte(signature))
	if err != nil {
		log.Println(err)
	}
	uid := []byte("tk")
	if sm2.Sm2Verify(&privateKeyBytes.PublicKey, []byte(creq.Request.Request), uid, r, s) {
		tx, err := sm2.Decrypt(privateKeyBytes, []byte(creq.Request.Request), sm2.C1C2C3)
		if err != nil {
			log.Println(err)
		}
		req.Head = creq.Request.Head
		err = json.Unmarshal(tx, req.Request)
		if err != nil {
			log.Println(err)
		}
		return req
	}

	return req
}
