#!/bin/bash

# 浦发农联体接口部署脚本
# 用法: ./deploy.sh [options]

set -e  # 遇到错误立即退出

# 配置变量
APP_NAME="pf"
PROD_PATH="/data/docker-compose/tk"
DOCKER_COMPOSE_FILE="docker-compose.yml"
BACKUP_DIR="/data/backup/nlt-pufa"

# 颜色输出
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# 打印带颜色的消息
print_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

print_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 显示帮助信息
show_help() {
    cat << EOF
浦发农联体接口部署脚本

用法: $0 [选项]

选项:
    -h, --help          显示此帮助信息
    -b, --backup        部署前备份当前版本
    -r, --restart-only  仅重启服务，不部署
    -l, --logs          查看服务日志
    -s, --status        查看服务状态
    --rollback          回滚到上一个版本

示例:
    $0                  # 标准部署
    $0 -b               # 备份后部署
    $0 -r               # 仅重启服务
    $0 -l               # 查看日志
EOF
}

# 检查依赖
check_dependencies() {
    print_info "检查部署环境..."
    
    # 检查Go环境
    if ! command -v go &> /dev/null; then
        print_error "Go环境未安装"
        exit 1
    fi
    
    # 检查Docker
    if ! command -v docker &> /dev/null; then
        print_error "Docker未安装"
        exit 1
    fi
    
    # 检查docker-compose
    if ! command -v docker-compose &> /dev/null; then
        print_error "docker-compose未安装"
        exit 1
    fi
    
    # 检查生产目录
    if [ ! -d "$PROD_PATH" ]; then
        print_error "生产目录不存在: $PROD_PATH"
        exit 1
    fi
    
    print_info "环境检查通过"
}

# 备份当前版本
backup_current() {
    if [ -f "$PROD_PATH/$APP_NAME" ]; then
        print_info "备份当前版本..."
        
        # 创建备份目录
        mkdir -p "$BACKUP_DIR"
        
        # 生成备份文件名（包含时间戳）
        BACKUP_FILE="$BACKUP_DIR/${APP_NAME}_$(date +%Y%m%d_%H%M%S).tar.gz"
        
        # 打包备份
        tar -czf "$BACKUP_FILE" -C "$PROD_PATH" "$APP_NAME" html/
        
        print_info "备份完成: $BACKUP_FILE"
    else
        print_warn "未找到当前版本，跳过备份"
    fi
}

# 编译应用
build_app() {
    print_info "编译应用程序..."
    
    # 交叉编译Linux版本
    CGO_ENABLED=0 GOOS=linux GOARCH=amd64 go build -o "$APP_NAME" .
    
    if [ ! -f "$APP_NAME" ]; then
        print_error "编译失败"
        exit 1
    fi
    
    print_info "编译完成"
}

# 部署应用
deploy_app() {
    print_info "部署应用到生产环境..."
    
    # 复制可执行文件
    cp "$APP_NAME" "$PROD_PATH/"
    
    # 复制静态文件
    if [ -d "html" ]; then
        cp -r html "$PROD_PATH/"
    fi
    
    # 设置执行权限
    chmod +x "$PROD_PATH/$APP_NAME"
    
    print_info "文件复制完成"
}

# 重启服务
restart_service() {
    print_info "重启Docker服务..."
    
    cd "$PROD_PATH"
    
    # 停止服务
    docker-compose -f "$DOCKER_COMPOSE_FILE" stop tkservice
    
    # 启动服务
    docker-compose -f "$DOCKER_COMPOSE_FILE" up -d tkservice
    
    print_info "服务重启完成"
}

# 检查服务状态
check_service() {
    print_info "检查服务状态..."
    
    cd "$PROD_PATH"
    docker-compose -f "$DOCKER_COMPOSE_FILE" ps tkservice
    
    # 等待服务启动
    sleep 5
    
    # 检查健康状态
    if curl -f http://localhost:8970/ > /dev/null 2>&1; then
        print_info "服务运行正常"
    else
        print_warn "服务可能未正常启动，请检查日志"
    fi
}

# 查看日志
show_logs() {
    cd "$PROD_PATH"
    docker-compose -f "$DOCKER_COMPOSE_FILE" logs -f tkservice
}

# 回滚到上一版本
rollback() {
    print_info "查找最新备份..."
    
    if [ ! -d "$BACKUP_DIR" ]; then
        print_error "备份目录不存在: $BACKUP_DIR"
        exit 1
    fi
    
    # 找到最新的备份文件
    LATEST_BACKUP=$(ls -t "$BACKUP_DIR"/${APP_NAME}_*.tar.gz 2>/dev/null | head -n1)
    
    if [ -z "$LATEST_BACKUP" ]; then
        print_error "未找到备份文件"
        exit 1
    fi
    
    print_info "回滚到: $LATEST_BACKUP"
    
    # 解压备份
    tar -xzf "$LATEST_BACKUP" -C "$PROD_PATH"
    
    # 重启服务
    restart_service
    
    print_info "回滚完成"
}

# 主函数
main() {
    case "${1:-deploy}" in
        -h|--help)
            show_help
            ;;
        -l|--logs)
            show_logs
            ;;
        -s|--status)
            cd "$PROD_PATH"
            docker-compose -f "$DOCKER_COMPOSE_FILE" ps
            ;;
        -r|--restart-only)
            check_dependencies
            restart_service
            check_service
            ;;
        --rollback)
            check_dependencies
            rollback
            check_service
            ;;
        -b|--backup|deploy)
            check_dependencies
            
            # 如果指定了备份选项
            if [ "$1" = "-b" ] || [ "$1" = "--backup" ]; then
                backup_current
            fi
            
            build_app
            deploy_app
            restart_service
            check_service
            
            print_info "部署完成！"
            ;;
        *)
            print_error "未知选项: $1"
            show_help
            exit 1
            ;;
    esac
}

# 执行主函数
main "$@" 