# 🚀 项目部署运行指南

## 📋 环境要求

### Go环境
- **Go版本**: `1.21.13+`
- **操作系统**: Linux/macOS/Windows
- **内存**: 建议 2GB+
- **存储**: 建议 1GB+ 可用空间

### 数据库环境
- **数据库**: MySQL 5.7+
- **数据库名**: `pr_order` 
- **核心数据表**: `loan_orderinfo`
- **字符集**: UTF-8
- **网络**: 确保应用可访问数据库

### 网络要求
- **应用端口**: `8970` (HTTP服务)
- **数据库端口**: `3306` (MySQL默认)

## 🔧 快速启动

### 1. 环境检查
```bash
# 检查Go版本
go version
# 输出应显示: go version go1.21.13 或更高版本

# 检查Git（用于代码下载）
git --version

# 检查MySQL连接
mysql -h数据库主机 -u用户名 -p数据库名
```

### 2. 代码部署
```bash
# 克隆项目（如果需要）
git clone <repository-url>

# 进入项目目录
cd nlt-pufa-interface

# 查看项目结构
ls -la
```

### 3. 依赖安装
```bash
# 下载Go模块依赖
go mod download

# 验证依赖完整性
go mod verify

# 查看依赖列表
go list -m all
```

### 4. 数据库配置

#### 当前配置
文件: `db/init.go`
```go
var dsn = `demeter:ajc4crb.buj-GHJ!pzw@tcp(10.0.0.2:3306)/pr_order?useUnicode=true&characterEncoding=utf8&useSSL=false`
```

#### 生产环境配置
**重要**: 修改为您的实际数据库信息
```go
var dsn = "用户名:密码@tcp(主机:端口)/数据库名?charset=utf8&parseTime=True&loc=Local"
```

#### 配置示例
```go
// 本地开发环境
var dsn = "root:password@tcp(127.0.0.1:3306)/pr_order?charset=utf8&parseTime=True&loc=Local"

// 生产环境
var dsn = "prod_user:prod_pass@tcp(192.168.1.100:3306)/pr_order?charset=utf8&parseTime=True&loc=Local"
```

### 5. 启动服务

#### 开发模式
```bash
# 直接运行（适合开发调试）
go run main.go

# 输出日志示例:
# 2024/01/01 12:00:00 Server listening on :8970
```

#### 生产模式
```bash
# 编译为可执行文件
go build -o nlt-pufa-interface main.go

# 运行可执行文件
./nlt-pufa-interface

# 后台运行
nohup ./nlt-pufa-interface > app.log 2>&1 &
```

### 6. 服务验证
```bash
# 检查服务状态
curl http://localhost:8970/
# 应该返回担保费缴纳HTML页面

# 检查进程
ps aux | grep nlt-pufa-interface

# 检查端口监听
netstat -tulpn | grep :8970
# 或使用lsof
lsof -i :8970
```

## 🏗️ 详细架构说明

### 启动流程图
```mermaid
%%{init: {'theme':'base', 'themeVariables': {'background': '#ffffff', 'primaryColor': '#fff', 'primaryTextColor': '#000', 'primaryBorderColor': '#000', 'lineColor': '#000'}}}%%
graph TD
    A[启动main.go] --> B[初始化Hertz服务器]
    B --> C[配置监听端口:8970]
    C --> D[注册API路由]
    D --> E[数据库连接初始化]
    E --> F[设置静态资源路由]
    F --> G[启动HTTP服务监听]
    G --> H[等待客户端请求]
    
    I[数据库连接失败] --> J[程序异常退出]
    E --> I
    
    style A fill:#e3f2fd,stroke:#1976d2
    style G fill:#c8e6c9,stroke:#388e3c
    style H fill:#fff3e0,stroke:#f57c00
    style I fill:#ffebee,stroke:#d32f2f
    style J fill:#ffebee,stroke:#d32f2f
```

### 服务组件
| 组件 | 文件位置 | 作用 | 端口/配置 |
|------|----------|------|-----------|
| **HTTP服务器** | main.go | 处理API请求 | :8970 |
| **路由系统** | route/register.go | 接口路径映射 | 3个POST接口 |
| **数据库连接** | db/init.go | 数据持久化 | MySQL连接池 |
| **业务处理器** | handler/*.go | 业务逻辑 | 认证、支付查询等 |
| **数据模型** | model/*.go | 数据结构和加密 | SM2加密 |
| **常量配置** | nltconst/*.go | 系统常量 | 错误码、密钥等 |
| **静态资源** | html/ | 前端页面 | 担保费页面 |

### 启动代码解析
```go
func main() {
    // 1. 创建Hertz服务器实例
    h := server.Default(
        server.WithHostPorts(":8970"),  // 监听8970端口
    )
    
    // 2. 注册业务API路由
    route.Register(h)  // 注册3个POST接口
    
    // 3. 静态资源路由（担保费页面）
    h.GET("/", func(c context.Context, ctx *app.RequestContext) {
        ctx.HTML(200, "index.html", nil)
    })
    
    // 4. 启动服务并阻塞等待
    h.Spin()  // 开始监听请求
}
```

## 🌐 可访问的接口

### 服务地址
- **基础URL**: `http://localhost:8970`
- **协议**: HTTP
- **内容类型**: JSON (API接口)

### 路由清单

| 方法 | 路径 | 功能 | 加密 | 用途 |
|------|------|------|------|------|
| **GET** | `/` | 担保费缴纳页面 | ❌ | 前端界面 |
| **POST** | `/nlt/pufa/auth` | 身份认证接口 | ✅ SM2 | 用户认证和贷款申请 |
| **POST** | `/nlt/pufa/queryPayResult` | 支付状态查询 | ✅ SM2 | 查询担保费支付状态 |
| **POST** | `/nlt/pufa/queryJumpUrl` | 跳转URL查询 | ❌ | 获取业务跳转链接 |

### 接口访问示例
```bash
# 1. 访问首页
curl http://localhost:8970/

# 2. 测试URL查询接口（无加密）
curl -X POST http://localhost:8970/nlt/pufa/queryJumpUrl \
  -H "Content-Type: application/json" \
  -d '{"duebillNoOrg":"test123"}'

# 3. 测试加密接口（需要SM2加密，较复杂）
# 参考 docs/API_INTEGRATION.md 了解加密调用方式
```

## 🐛 开发调试

### 热重载开发
```bash
# 安装air工具
go install github.com/cosmtrek/air@latest

# 创建.air.toml配置文件
cat > .air.toml << EOF
root = "."
testdata_dir = "testdata"
tmp_dir = "tmp"

[build]
bin = "./tmp/main"
cmd = "go build -o ./tmp/main ."
delay = 1000
exclude_dir = ["assets", "tmp", "vendor", "testdata"]
exclude_file = []
exclude_regex = ["_test.go"]
exclude_unchanged = false
follow_symlink = false
full_bin = ""
include_dir = []
include_ext = ["go", "tpl", "tmpl", "html"]
kill_delay = "0s"
log = "build-errors.log"
stop_on_root = false

[color]
app = ""
build = "yellow"
main = "magenta"
runner = "green"
watcher = "cyan"

[log]
time = false

[misc]
clean_on_exit = false
EOF

# 启动热重载
air
```

### 日志调试
```bash
# 启用详细日志
export DEBUG=true
go run main.go

# 查看特定模块日志
export LOG_LEVEL=debug
go run main.go

# 重定向日志到文件
go run main.go > app.log 2>&1
```

### 性能分析
```bash
# 启用性能分析
go run main.go -cpuprofile=cpu.prof -memprofile=mem.prof

# 分析CPU使用
go tool pprof cpu.prof

# 分析内存使用
go tool pprof mem.prof
```

## 📊 运行监控

### 健康检查脚本
```bash
#!/bin/bash
# health_check.sh

# 检查服务是否运行
check_service() {
    local response=$(curl -s -o /dev/null -w "%{http_code}" http://localhost:8970/)
    if [ "$response" = "200" ]; then
        echo "✅ 服务运行正常"
        return 0
    else
        echo "❌ 服务异常，HTTP状态码: $response"
        return 1
    fi
}

# 检查数据库连接
check_database() {
    local response=$(curl -s -X POST http://localhost:8970/nlt/pufa/queryJumpUrl \
        -H "Content-Type: application/json" \
        -d '{"duebillNoOrg":"health_check"}')
    
    if echo "$response" | grep -q "head"; then
        echo "✅ 数据库连接正常"
        return 0
    else
        echo "❌ 数据库连接异常"
        return 1
    fi
}

# 执行检查
echo "🔍 开始健康检查..."
check_service && check_database
echo "✨ 健康检查完成"
```

### 系统资源监控
```bash
# 监控进程资源使用
top -p $(pgrep -f nlt-pufa-interface)

# 监控内存使用
ps aux | grep nlt-pufa-interface

# 监控网络连接
netstat -tulpn | grep :8970

# 监控日志输出
tail -f app.log
```

### 性能指标
```bash
# CPU使用率
top -p $(pgrep -f nlt-pufa-interface) -n 1 | grep nlt-pufa

# 内存使用
ps -o pid,ppid,cmd,%mem,%cpu --sort=-%mem | grep nlt-pufa

# 网络连接数
netstat -an | grep :8970 | wc -l

# 磁盘IO
iotop -p $(pgrep -f nlt-pufa-interface)
```

## ⚠️ 注意事项

### 安全配置
1. **数据库密码**: 生产环境请使用强密码
2. **端口访问**: 配置防火墙限制8970端口访问
3. **SSL证书**: 生产环境建议配置HTTPS
4. **密钥管理**: SM2密钥请妥善保管

### 运维建议
1. **备份策略**: 定期备份数据库和配置文件
2. **日志轮转**: 配置日志轮转避免磁盘空间不足
3. **监控告警**: 配置服务和数据库监控告警
4. **版本管理**: 使用Git标签管理版本发布

### 故障排查
1. **服务无法启动**: 
   - 检查端口是否被占用
   - 检查数据库连接配置
   - 查看启动日志错误信息

2. **数据库连接失败**:
   - 验证数据库服务是否运行
   - 检查网络连通性
   - 确认用户名密码正确

3. **接口调用异常**:
   - 检查请求格式是否正确
   - 验证SM2加密是否正确
   - 查看服务端日志

## 🔧 生产环境部署

### 实际生产环境配置

本项目在生产环境中使用Docker容器化部署，具体配置如下：

#### 生产环境架构
- **服务器路径**: `/data/docker-compose/tk`
- **容器名称**: `tkservice`
- **基础镜像**: `alios:v3.0.1` (自定义阿里云OS镜像)
- **可执行文件**: `pf` (编译后的二进制文件)
- **网络**: `my_network` (外部网络)
- **日志文件**: `pf.log`

#### 实际的 docker-compose.yml 配置

```yaml
# /data/docker-compose/tk/docker-compose.yml
version: '3'

services:
  tkservice:
    image: alios:v3.0.1
    container_name: tkservice
    restart: always
    ports:
      - 8970:8970
    environment:
      - TZ=Asia/Shanghai
    volumes:
      - /data/docker-compose/tk:/home/tk
    working_dir: /home/tk
    command: ./pf >> /home/tk/pf.log

networks:
  default:
    external: true
    name: my_network
```

### 部署流程

#### 使用部署脚本（推荐）

项目提供了自动化部署脚本 `deploy.sh`：

```bash
# 标准部署
./deploy.sh

# 备份后部署
./deploy.sh -b

# 仅重启服务
./deploy.sh -r

# 查看服务状态
./deploy.sh -s

# 查看日志
./deploy.sh -l

# 回滚到上一版本
./deploy.sh --rollback
```

#### 使用 Makefile

```bash
# 编译生产版本
make build-linux

# 部署到生产环境
make deploy

# 重启生产服务
make restart

# 查看生产日志
make logs

# 查看服务状态
make status
```

#### 手动部署步骤

1. **编译应用**：
```bash
# 交叉编译Linux版本
CGO_ENABLED=0 GOOS=linux GOARCH=amd64 go build -o pf .
```

2. **部署文件**：
```bash
# 复制到生产目录
cp pf /data/docker-compose/tk/
cp -r html /data/docker-compose/tk/
chmod +x /data/docker-compose/tk/pf
```

3. **重启服务**：
```bash
cd /data/docker-compose/tk
docker-compose restart tkservice
```

### 生产环境监控

#### 服务状态检查
```bash
# 检查容器状态
docker ps | grep tkservice

# 检查服务详情
cd /data/docker-compose/tk
docker-compose ps

# 检查端口监听
netstat -tlnp | grep 8970
```

#### 日志监控
```bash
# 实时查看日志
cd /data/docker-compose/tk
tail -f pf.log

# 查看Docker容器日志
docker-compose logs -f tkservice

# 日志分析
grep "ERROR" pf.log
grep "panic" pf.log
```

#### 健康检查脚本
```bash
#!/bin/bash
# health_check.sh

ENDPOINT="http://localhost:8970/"
TIMEOUT=5

if curl -f --connect-timeout $TIMEOUT "$ENDPOINT" > /dev/null 2>&1; then
    echo "✅ 服务运行正常"
    exit 0
else
    echo "❌ 服务异常，请检查日志"
    exit 1
fi
```

### 备份与恢复

#### 自动备份
部署脚本提供自动备份功能：
```bash
# 部署前自动备份
./deploy.sh -b
```

备份文件存储在 `/data/backup/nlt-pufa/` 目录下。

#### 手动备份
```bash
# 创建备份目录
mkdir -p /data/backup/nlt-pufa

# 备份当前版本
cd /data/docker-compose/tk
tar -czf "/data/backup/nlt-pufa/pf_$(date +%Y%m%d_%H%M%S).tar.gz" pf html/
```

#### 恢复操作
```bash
# 自动回滚到最新备份
./deploy.sh --rollback

# 手动恢复指定备份
cd /data/docker-compose/tk
tar -xzf /data/backup/nlt-pufa/pf_20240101_120000.tar.gz
docker-compose restart tkservice
```

---

**文档版本**: v1.0  
**最后更新**: 2024年  
**维护团队**: 普发银行农业贷款系统运维组 