package model

import (
	"crypto/rand"
	"encoding/hex"
	"encoding/json"
	"errors"
	"fmt"
	"log"
	"math/big"

	"github.com/tjfoc/gmsm/sm2"
	"nlt.com/pf/nltconst"
)

type CryptHttpBodyReq struct {
	Request   HttpBodyReq[string] `json:"request"`
	Signature string              `json:"signature"`
}

type CryptHttpBodyResp struct {
	Response  HttpBodyResp[string] `json:"response"`
	Signature string               `json:"signature"`
}

type HttpBodyReq[T any] struct {
	Head    ReqHead `json:"head"`
	Request T       `json:"body"`
}

type HttpBodyResp[T any] struct {
	Head     RespHead `json:"head"`
	Response T        `json:"body"`
}

type ReqHead struct {
	RequestTime string `json:"requestTime"`
	ServiceSn   string `json:"serviceSn"`
}

type RespHead struct {
	Code        string `json:"code"`
	ServiceTime string `json:"serviceTime"`
	ServiceSn   string `json:"serviceSn"`
}

func hexToPrivateKey(hexPrivKey string) (*sm2.PrivateKey, error) {
	// 解码Hex字符串
	privKeyBytes, err := hex.DecodeString(hexPrivKey)
	if err != nil {
		return nil, err
	}

	// 创建私钥对象
	privateKey := new(sm2.PrivateKey)
	privateKey.PublicKey.Curve = sm2.P256Sm2()
	privateKey.D = new(big.Int).SetBytes(privKeyBytes)
	// 计算公钥
	privateKey.PublicKey.X, privateKey.PublicKey.Y = privateKey.PublicKey.Curve.ScalarBaseMult(privKeyBytes)

	// 验证私钥是否有效
	if !privateKey.PublicKey.Curve.IsOnCurve(privateKey.PublicKey.X, privateKey.PublicKey.Y) {
		return nil, fmt.Errorf("private key not valid")
	}

	return privateKey, nil
}

func EncryptAndSign[T any](resp HttpBodyResp[T]) (CryptHttpBodyResp, error) {
	var cresp CryptHttpBodyResp
	privateKeyBytes, _ := hexToPrivateKey(nltconst.SM2_PRIVATE_KEY)

	// 对应的公钥
	privateBankey, err := hexToPrivateKey(nltconst.TEST_BANK_PRIVTE_KEY)
	if err != nil {
		log.Println(err)
		return cresp, err
	}
	publicKey := privateBankey.PublicKey

	body, err := json.Marshal(resp.Response)
	if err != nil {
		log.Println(err.Error())
		return cresp, err
	}

	ciphertext, err := sm2.Encrypt(&publicKey, body, rand.Reader, sm2.C1C3C2)
	if err != nil {
		log.Println(err)
		return cresp, err
	}
	cresp.Response.Head = resp.Head
	cresp.Response.Response = hex.EncodeToString(ciphertext)

	jsonResp, err := json.Marshal(cresp.Response)
	if err != nil {
		log.Println(err)
		return cresp, err
	}
	fmt.Println(string(jsonResp))
	signature, err := privateKeyBytes.Sign(rand.Reader, jsonResp, nil)

	cresp.Signature = hex.EncodeToString(signature)
	fmt.Println(cresp.Signature)
	return cresp, err
}

func VerifyAndDecrypt[T any](creq CryptHttpBodyReq) (HttpBodyReq[T], error) {
	var req HttpBodyReq[T]
	privateKeyBytes, _ := hexToPrivateKey(nltconst.SM2_PRIVATE_KEY)

	// 对应的公钥
	privateBankey, err := hexToPrivateKey(nltconst.TEST_BANK_PRIVTE_KEY)
	if err != nil {
		log.Println(err)
		return req, err
	}
	publicKey := privateBankey.PublicKey
	signature, err := hex.DecodeString(creq.Signature)
	if err != nil {
		log.Println(err)
		return req, err
	}

	jsonReq, err := json.Marshal(creq.Request)
	fmt.Println(creq.Request)
	if err != nil {
		log.Println(err)
		return req, err
	}

	if publicKey.Verify(jsonReq, signature) {
		body, err := hex.DecodeString(creq.Request.Request)
		if err != nil {
			log.Println(err)
			return req, errors.New("解密错误")
		}
		tx, err := sm2.Decrypt(privateKeyBytes, body, sm2.C1C3C2)
		if err != nil {
			log.Println(err)
			return req, errors.New("解密错误")
		}
		req.Head = creq.Request.Head
		err = json.Unmarshal(tx, &req.Request)
		if err != nil {
			log.Println(err)
			return req, err
		}
		return req, err
	} else {
		return req, errors.New("验签错误")
	}
}
