package model

import (
	"crypto/rand"
	"encoding/base64"
	"encoding/hex"
	"encoding/json"
	"errors"
	"log"
	"strings"

	"github.com/tjfoc/gmsm/sm2"
	"nlt.com/pf/nltconst"
)

type CryptHttpBodyReq struct {
	Request   HttpBodyReq[string]
	Signature string
}

type CryptHttpBodyResp struct {
	Response  HttpBodyResp[string]
	Signature string
}

type HttpBodyReq[T any] struct {
	Head    ReqHead `json:"head"`
	Request T       `json:"body"`
}

type HttpBodyResp[T any] struct {
	Head     RespHead `json:"head"`
	Response T        `json:"body"`
}

type ReqHead struct {
	RequestTime string `json:"requestTime"`
	ServiceSn   string `json:"serviceSn"`
}

type RespHead struct {
	Code        string `json:"code"`
	ServiceTime string `json:"serviceTime"`
	ServiceSn   string `json:"serviceSn"`
}

func EncryptAndSign[T any](resp HttpBodyResp[T]) (CryptHttpBodyResp, error) {
	var cresp CryptHttpBodyResp
	privateKeyBytes, _ := sm2.GenerateKey(strings.NewReader(nltconst.SM2_PRIVATE_KEY))

	// 对应的公钥
	publicKeyBytes, err := hex.DecodeString(nltconst.BANK_PUBLIC_KEY)
	if err != nil {
		log.Println(err.Error())
		return cresp, err
	}

	publicKey := sm2.Decompress(publicKeyBytes)
	body, err := json.Marshal(resp.Response)
	if err != nil {
		log.Println(err.Error())
		return cresp, err
	}
	ciphertext, err := sm2.Encrypt(publicKey, body, rand.Reader, sm2.C1C2C3)
	if err != nil {
		log.Println(err)
		return cresp, err
	}
	cresp.Response.Head = resp.Head
	cresp.Response.Response = base64.RawStdEncoding.EncodeToString(ciphertext)

	jsonResp, err := json.Marshal(resp)
	if err != nil {
		log.Println(err)
		return cresp, err
	}
	uid := []byte("tk")
	r, s, err := sm2.Sm2Sign(privateKeyBytes, jsonResp, uid, rand.Reader)
	if err != nil {
		log.Println(err)
		return cresp, err
	}
	signature, err := sm2.SignDigitToSignData(r, s)
	if err != nil {
		log.Println(err)
		return cresp, err
	}

	cresp.Signature = base64.RawStdEncoding.EncodeToString(signature)
	return cresp, err
}

func VerifyAndDecrypt[T any](creq CryptHttpBodyReq) (HttpBodyReq[T], error) {
	var req HttpBodyReq[T]
	privateKeyBytes, _ := sm2.GenerateKey(strings.NewReader(nltconst.SM2_PRIVATE_KEY))

	// 对应的公钥
	publicKeyBytes, err := hex.DecodeString(nltconst.BANK_PUBLIC_KEY)
	if err != nil {
		log.Println(err.Error())
	}
	publicKey := sm2.Decompress(publicKeyBytes)

	signature, err := base64.RawStdEncoding.DecodeString(creq.Signature)
	if err != nil {
		log.Println(err)
		return req, err
	}
	r, s, err := sm2.SignDataToSignDigit(signature)
	if err != nil {
		log.Println(err)
		return req, err
	}
	uid := []byte("tk")
	if sm2.Sm2Verify(publicKey, signature, uid, r, s) {
		body, err := base64.RawStdEncoding.DecodeString(creq.Request.Request)
		if err != nil {
			log.Println(err)
			return req, errors.New("解密错误")
		}
		tx, err := sm2.Decrypt(privateKeyBytes, body, sm2.C1C2C3)
		if err != nil {
			log.Println(err)
			return req, errors.New("解密错误")
		}
		req.Head = creq.Request.Head
		err = json.Unmarshal(tx, req.Request)
		if err != nil {
			log.Println(err)
			return req, err
		}
		return req, err
	} else {
		return req, errors.New("验签错误")
	}
}
